﻿<#
.SYNOPSIS
 Demo of creating a ConfigItem with multiple settings and rules. Written for Midwest Management Summit 2019, mmsmoa2019.

.DESCRIPTION
 Using ConfigMgr Powershell CMDlets, create a CI.

.Notes
 Leveraged https://blog.ctglobalservices.com/uncategorized/rja/create-configurationitems-and-baselines-without-killing-your-mouse/ for code and logic
 You will need to at a minimum
   - Modify SiteCode and ProviderMachineName to match your environment
   - Run this script from a machine which has the Configuration Manager Powershell CMDlets available (in general, if the admin console is installed, the cmdlets are available)
   - Run this script using Security Context of an account which has rights to the CM Environment you referenced in the SiteCode / ProviderMachineName

 When script is done, a single Configuration Item with the name you have in the variable $CIName will have been created.  
    That CI will have multiple Settings inside, with 1 rule per setting, remediation possible, and old revisions removed.

 Also note that for demo purposes, if a CI already exists with the name contained in $CIName, it will be deleted first, then then recreated via this script.

 2019-03-20 v1.0 Sherry Kissinger

.Lessons Learned while writing this script
 - when using @"   
   "@
   for passing code, if WITHIN that code you want to have a variable, you need to add `  before the variable so it knows it needs to be literal, not replacing.
   
  
#>
$ErrorActionPreference = 'Continue'
$VerbosePreference = 'Continue'
$SiteCode = "ZND" # Site code 
$ProviderMachineName = "hs2.mofmistress.local" # SMS Provider machine name
$CIName = "MMS2019 Demo WSUS"
$DescriptionOftheCIItself = "MMSMOA 2019 Testing"

$SettingName1 = "WSUS Administration Max Connections Should be Unlimited"
$RuleName1 = "ShouldbeUnlimited-4294967295"
$ResultType1 = "Integer"
$ExpectedValue1 = 4294967295
$Detect1 = "import-Module webadministration ; (get-itemproperty IIS:\Sites\'WSUS Administration' -name limits.maxConnections.Value)"
$Remediate1 = @"
#Note, that CaSe is important. QueueLength <> queueLength.  Has to be queueLength in order to be set
import-Module webadministration ; 
set-Itemproperty IIS:\Sites\'WSUS Administration' -Name limits.maxConnections -Value 4294967295
"@

$SettingName2 = "WSUS Administration MaxBandwidth should be unlimited"
$RuleName2 = 'MazBandwidthShouldbe-Unlimited-4294967295'
$ResultType2 = "Integer"
$ExpectedValue2 = 4294967295
$Detect2 = "import-Module webadministration ; (get-itemproperty IIS:\Sites\'WSUS Administration' -name limits.maxbandwidth.Value)"
$Remediate2 = @"
#Note, that CaSe is important. QueueLength <> queueLength.  Has to be queueLength in order to be set
import-Module webadministration
set-Itemproperty IIS:\Sites\'WSUS Administration' -Name limits.maxBandwidth -Value 4294967295
"@

$SettingName3 = "WSUS Administration TimeOut should be 320"
$RuleName3 = 'TImeoutShould be 320'
$ResultType3 = "Integer"
$ExpectedValue3 = 320
$Detect3 = "import-Module webadministration ; (get-itemproperty IIS:\Sites\'WSUS Administration' -Name limits.connectionTimeout.value).TotalSeconds"
$Remediate3 = @"
#Note, that CaSe is important. QueueLength <> queueLength.  Has to be queueLength in order to be set
import-Module webadministration ; set-Itemproperty IIS:\Sites\'WSUS Administration' -Name limits.connectionTimeout -Value 00:05:20
"@

$SettingName4 = "WSUS Service Should be Running"
$RuleName4 = 'wsusservice Should be Running'
$ResultType4 = "String"
$ExpectedValue4 = 'Running'
$Detect4 =@"
(Get-Service "WSUS Service").Status
"@
$Remediate4 = @"
Start-Service "WSUS Service"
"@

$SettingName5 = "WSUSPool Application Pool Should be Started"
$RuleName5 = 'wsuspool Should be Started'
$ResultType5 = "String"
$ExpectedValue5 = 'Started'
$Detect5 ='import-Module webadministration ; (Get-WebAppPoolState WSUSPool).Value'
$Remediate5 = @"
import-Module webadministration
Start-WebAppPool -Name "WSUSPool"
"@

$SettingName6 = "WSUSPool CPU ResetInterval should be 15 min"
$RuleName6 = 'wsuspool cpu Should be 15'
$ResultType6 = "Integer"
$ExpectedValue6 = 15
$Detect6 = @"
import-Module webadministration ; (get-itemproperty IIS:\AppPools\Wsuspool -Name cpu.resetInterval.value).minutes
"@
$Remediate6 = @"
#Note, that CaSe is important. QueueLength <> queueLength.  Has to be queueLength in order to be set
import-Module webadministration ; set-Itemproperty IIS:\AppPools\Wsuspool -Name cpu -Value @{resetInterval="00:15:00"}
"@

$SettingName7 = "WSUSPool Ping Disabled"
$RuleName7 = 'wsuspool ping Should be False'
$ResultType7 = "String"
$ExpectedValue7 = "False"
$Detect7 = @"
import-Module webadministration
(get-itemproperty IIS:\AppPools\Wsuspool -Name processmodel.pingingEnabled).value
"@
$Remediate7 = @"
#Note, that CaSe is important. QueueLength <> queueLength.  Has to be queueLength in order to be set
import-Module webadministration
set-Itemproperty IIS:\AppPools\Wsuspool -Name processmodel.pingingEnabled False
"@

$SettingName8 = "WSUSPool queueLength should be 30000"
$RuleName8 = 'wsuspool queueLength should be 30000'
$ResultType8 = "Integer"
$ExpectedValue8 = 30000
$Detect8 = @"
import-Module webadministration ; (get-itemproperty IIS:\AppPools\Wsuspool | Select queuelength).queueLength
"@
$Remediate8 = @"
#Note, that CaSe is important. QueueLength <> queueLength.  Has to be queueLength in order to be set
import-Module webadministration ; set-Itemproperty IIS:\AppPools\Wsuspool -name queueLength 30000
"@

$SettingName9 = "WSUSPool RapidFail Should be Disabled"
$RuleName9 = 'wsuspool RapidFail'
$ResultType9 = "String"
$ExpectedValue9 = "False"
$Detect9 = @"
import-Module webadministration
(get-itemproperty IIS:\AppPools\Wsuspool -name failure.rapidFailProtection).Value
"@
$Remediate9 = @"
import-Module webadministration
set-Itemproperty IIS:\AppPools\Wsuspool -name failure.rapidFailProtection False
"@

$SettingName10 = "WSUSPool Recycling Regular Time interval should be 0"
$RuleName10 = 'wsuspool recycling Should be 0'
$ResultType10 = "Integer"
$ExpectedValue10 = 0
$Detect10= @"
import-Module webadministration ; ((get-itemproperty IIS:\AppPools\Wsuspool -name recycling.periodicRestart.time).Value).TotalMinutes
"@
$Remediate10 = @"
#Note, that CaSe is important. QueueLength <> queueLength.  Has to be queueLength in order to be set
import-Module webadministration ; set-Itemproperty IIS:\AppPools\Wsuspool recycling.periodicRestart.time -Value 00:00:00
"@

$SettingName11 = "WSUSPool Requests Should be 0"
$RuleName11 = 'wsuspool requests Shouldbe0'
$ResultType11 = "Integer"
$ExpectedValue11 = 0
$Detect11 = @"
import-module webadministration
(Get-WebConfiguration "/system.applicationHost/applicationPools/add[@name='WsusPool']/recycling/periodicRestart/@requests").Value
"@
$Remediate11 = @"
import-module webadministration
Set-WebConfiguration "/system.applicationHost/applicationPools/add[@name='WsusPool']/recycling/periodicRestart/@requests" -Value 0
"@

$SettingName12 = "WSUSPool Private Memory Limit should be 0"
$RuleName12 = "wsuspool pml Shouldbe0"
$ResultType12 = "Integer"
$ExpectedValue12 = 0
$Detect12 = @"
import-module webadministration
(Get-WebConfiguration "/system.applicationHost/applicationPools/add[@name='WsusPool']/recycling/periodicRestart/@privateMemory").Value
"@
$Remediate12 = @"
import-module webadministration
Set-WebConfiguration "/system.applicationHost/applicationPools/add[@name='WsusPool']/recycling/periodicRestart/@privateMemory" -Value 0
"@

$SettingName13 = "WSUS ClientWebService web.config maxRequestLength should be 20480"
$RuleName13 = "wsus clientwebservice Should be 20480"
$ResultType13 = "Integer"
$ExpectedValue13 = 20480
$Detect13 = @"
import-Module webadministration
[XML]`$xml = Get-Content (Get-WebConfigFile 'IIS:\Sites\WSUS Administration\ClientWebService').fullname
(((`$xml.configuration).'system.web').httpRunTime).maxRequestLength
"@
$Remediate13 = @"
import-Module webadministration
`$FullFileName = (Get-WebConfigFile 'IIS:\Sites\WSUS Administration\ClientWebService').fullname
`$acl = get-acl `$FullFileName
`$Ar = New-Object system.security.accesscontrol.filesystemaccessrule("NT AUTHORITY\SYSTEM","FullControl","Allow")
`$acl.SetAccessRule($Ar)
Set-ACL `$FullFileName `$acl
[XML]`$xml = Get-Content `$FullFileName
`$ChangeThis = (((`$xml.configuration).'system.web').httpRunTime)
`$ChangeThis.maxRequestLength = "20480"
`$xml.Save(`$FullFileName)
"@

$SettingName14 = "WSUS ClientWebService web.config executionTimeout should be 7200"
$RuleName14 = "clientwebservice Should be 7200"
$ResultType14 = "Integer"
$ExpectedValue14 = 7200
$Detect14 = @"
import-Module webadministration
`$FullFileName = (Get-WebConfigFile 'IIS:\Sites\WSUS Administration\ClientWebService').fullname
[XML]`$xml = Get-Content `$FullFileName
(((`$xml.configuration).'system.web').httpRunTime).executionTimeout
"@
$Remediate14 = @"
import-Module webadministration
`$FullFileName = (Get-WebConfigFile 'IIS:\Sites\WSUS Administration\ClientWebService').fullname
`$acl = get-acl `$FullFileName
`$Ar = New-Object system.security.accesscontrol.filesystemaccessrule("NT AUTHORITY\SYSTEM","FullControl","Allow")
`$acl.SetAccessRule(`$Ar)
Set-ACL `$FullFileName `$acl
[XML]`$xml = Get-Content `$FullFileName
`$ChangeThis = (((`$xml.configuration).'system.web').httpRunTime)
`$ChangeThis.SetAttribute('executionTimeout','7200')
`$xml.Save(`$FullFileName)
"@

$SettingName15 = "WSUS Administration Logging Enable BytesSent"
$RuleName15 = "admin logging BytesSent Enabled"
$ResultType15 = "String"
$ExpectedValue15 = "BytesSent"
$Detect15 = @"
import-Module webadministration 
`$value = 'BytesSent'
`$ExistingValues = (get-itemproperty IIS:\Sites\'WSUS Administration' -name LogFile).logExtFileFlags
if (`$ExistingValues -match `$Value ) {
write-host `$Value
}
Else {write-host `$Value 'Not Found'}
"@
$Remediate15 = @"
import-Module webadministration 
`$value = 'BytesSent'
`$ExistingValues = (get-itemproperty IIS:\Sites\'WSUS Administration' -name LogFile).logExtFileFlags
if (`$ExistingValues -notmatch `$Value ) {
`$NewValue = `$ExistingValues + ',' + `$value
Set-ItemProperty 'IIS:\Sites\WSUS Administration' -Name logfile -Value @{logExtFileFlags = `$NewValue}
}
"@

$SettingName16 = "WSUS Administration Logging Enable BytesRecv"
$RuleName16 = "admin logging BytesRecv Enabled"
$ResultType16 = "String"
$ExpectedValue16 = "BytesRecv"
$Detect16 = @"
import-Module webadministration 
`$value = 'BytesRecv'
`$ExistingValues = (get-itemproperty IIS:\Sites\'WSUS Administration' -name LogFile).logExtFileFlags
if (`$ExistingValues -match `$Value ) {
write-host `$Value
}
Else {write-host `$Value 'Not Found'}
"@
$Remediate16 = @"
import-Module webadministration 
`$value = 'BytesRecv'
`$ExistingValues = (get-itemproperty IIS:\Sites\'WSUS Administration' -name LogFile).logExtFileFlags
if (`$ExistingValues -notmatch `$Value ) {
`$NewValue = `$ExistingValues + ',' + `$value
Set-ItemProperty 'IIS:\Sites\WSUS Administration' -Name logfile -Value @{logExtFileFlags = `$NewValue}
}
"@

$ApplicationDetectionLogic = @"
(Get-WmiObject -Namespace root\cimv2 -class win32_serverfeature -Filter "Name = 'WINDOWS SERVER UPDATE SERVICES'").Name
"@
 
Write-Verbose "Connect to CM, You may need to modify the SiteCode and Provider below"

# Customizations
$initParams = @{}
if((Get-Module ConfigurationManager) -eq $null) {
    Import-Module "$($ENV:SMS_ADMIN_UI_PATH)\..\ConfigurationManager.psd1" @initParams 
}
# Connect to the site's drive if it is not already present
if((Get-PSDrive -Name $SiteCode -PSProvider CMSite -ErrorAction SilentlyContinue) -eq $null) {
    New-PSDrive -Name $SiteCode -PSProvider CMSite -Root $ProviderMachineName @initParams
}
# Set the current location to be the site code.
Set-Location "$($SiteCode):\" @initParams
Write-Verbose "CM Connection Complete if possible"
Write-Verbose "Remove any existing baselines or Configuration Items with that CIName (paranoia step, during testing / for MMS.  Comment or Remove when you might use this for real)"
#Get-CMBaseline -Name $CIName | Remove-CMBaseline -Force
#------------------------------------------------------------------------------------------------------------
Get-CMConfigurationItem -Name $CIName -Fast | Remove-CMConfigurationItem -Force
#------------------------------------------------------------------------------------------------------------
Write-Verbose "Create a New Configuration Item, as a WindowsApplication type"
$ci = New-CMConfigurationItem -Name $CIName -CreationType WindowsApplication -Description $DescriptionOftheCIItself
#------------------------------------------------------------------------------------------------------------
<#
Yeah, yeah... I know I should have done some kind of for-next or splat or whatever here for the multiple rules...
maybe next time.  I just wanted to see it work at all.
#>
Write-Verbose "For the $ci just created, add the multiple settings and rules"

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType1 -DiscoveryScriptText $Detect1 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate1 -RemediationScriptLanguage PowerShell -SettingName $SettingName1 `
   -RuleName $RuleName1 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue1 

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType2 -DiscoveryScriptText $Detect2 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate2 -RemediationScriptLanguage PowerShell -SettingName $SettingName2 `
   -RuleName $RuleName2 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue2 

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType3 -DiscoveryScriptText $Detect3 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate3 -RemediationScriptLanguage PowerShell -SettingName $SettingName3 `
   -RuleName $RuleName3 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue3

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType4 -DiscoveryScriptText $Detect4 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate4 -RemediationScriptLanguage PowerShell -SettingName $SettingName4 `
   -RuleName $RuleName4 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue4
   
$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType5 -DiscoveryScriptText $Detect5 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate5 -RemediationScriptLanguage PowerShell -SettingName $SettingName5 `
   -RuleName $RuleName5 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue5

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType6 -DiscoveryScriptText $Detect6 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate6 -RemediationScriptLanguage PowerShell -SettingName $SettingName6 `
   -RuleName $RuleName6 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue6

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType7 -DiscoveryScriptText $Detect7 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate7 -RemediationScriptLanguage PowerShell -SettingName $SettingName7 `
   -RuleName $RuleName7 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue7
   
$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType8 -DiscoveryScriptText $Detect8 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate8 -RemediationScriptLanguage PowerShell -SettingName $SettingName8 `
   -RuleName $RuleName8 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue8

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType9 -DiscoveryScriptText $Detect9 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate9 -RemediationScriptLanguage PowerShell -SettingName $SettingName9 `
   -RuleName $RuleName9 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue9
   
$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType10 -DiscoveryScriptText $Detect10 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate10 -RemediationScriptLanguage PowerShell -SettingName $SettingName10 `
   -RuleName $RuleName10 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue10
   
$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType11 -DiscoveryScriptText $Detect11 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate11 -RemediationScriptLanguage PowerShell -SettingName $SettingName11 `
   -RuleName $RuleName11 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue11

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType12 -DiscoveryScriptText $Detect12 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate12 -RemediationScriptLanguage PowerShell -SettingName $SettingName12 `
   -RuleName $RuleName12 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue12

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType13 -DiscoveryScriptText $Detect13 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate13 -RemediationScriptLanguage PowerShell -SettingName $SettingName13 `
   -RuleName $RuleName13 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue13
   
$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType14 -DiscoveryScriptText $Detect14 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate14 -RemediationScriptLanguage PowerShell -SettingName $SettingName14 `
   -RuleName $RuleName14 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue14
   
$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType15 -DiscoveryScriptText $Detect15 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate15 -RemediationScriptLanguage PowerShell -SettingName $SettingName15 `
   -RuleName $RuleName15 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue15

$temp = $ci | Add-CMComplianceSettingScript -DataType $ResultType16 -DiscoveryScriptText $Detect16 -DiscoveryScriptLanguage PowerShell `
   -RemediationScriptText $Remediate16 -RemediationScriptLanguage PowerShell -SettingName $SettingName16 `
   -RuleName $RuleName16 -ValueRule -ExpressionOperator IsEquals -ExpectedValue $ExpectedValue16
#------------------------------------------------------------------------------------------------------------
Write-Verbose "For the $ci just created, add Application script for Applicability"
$CIDeserved = Get-CMConfigurationItem -Name $CIName | ConvertTo-CMConfigurationItem
$CIDeserved.DetectionMethod = "Script"
$CIDeserved.ScriptLanguage = "PowerShell"
$CIDeserved.CustomDetectionScript = $ApplicationDetectionLogic
Set-CMConfigurationItem -Name $CIName -Digest $CIDeserved
#------------------------------------------------------------------------------------------------------------
Write-Verbose "Change to 64-bit scripts, and also set the rules to remediate if non-compliant"
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName1
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName2
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName3
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName4
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName5
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName6
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName7
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName8
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName9
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName10
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName11
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName12
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName13
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName14
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName15
$ci | Set-CMComplianceSettingScript -Is64Bit $true -SettingName $SettingName16
$ci | Set-CMComplianceRuleValue -RuleName $RuleName1 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName2 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName3 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName4 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName5 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName6 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName7 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName8 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName9 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName10 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName11 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName12 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName13 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName14 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName15 -Remediate $true
$ci | Set-CMComplianceRuleValue -RuleName $RuleName16 -Remediate $true
#------------------------------------------------------------------------------------------------------------
Write-Verbose "Cleanup Revisions"
$CIRevisions = Get-CMConfigurationItem -Name $CIName -Fast | Get-CMConfigurationItemHistory
# Note to self... using $i to count only works because we just made it.
# if this were an old CI, where maybe we manually deleted revisions 1 through 3, out of 6... 
# there would be count=3... but the revision numbers would be 4,5,6 (not 1,2,3, which is what the script looks for now
for ($i =0;$i -lt $CIRevisions.Count;$i++) {
   $CheckIfExists = Get-CMConfigurationItemHistory -Name $CIName -Revision $i 
   if ($CheckIfExists) {
      $ModelName = $CheckIfExists.ModelName 
      $CIVersion = $CheckIfExists.CIVersion
      Get-WMIObject -ComputerName $ProviderMachineName -Class "SMS_ConfigurationItem" -NameSpace "ROOT\SMS\site_$SiteCode" -Filter "ModelName='$ModelName' AND CIVersion = $CIVersion" | Remove-WmiObject
      }
}

Write-Verbose "And we're done.  Because of creating, then adding a setting with a rule, and updating detectionmethod, the CI will have iterated multiple versions already"
